/** @file   animplayer.cpp
 * @brief   Implementation of AnimPlayer class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "animplayer.h"
#include "www_map.h"
#include "settings.h"
#include "soundsamples.h"

using std::list;
using namespace eng2d;


namespace WeWantWar {


const int AnimPlayer::MAX_ANIMS;
int AnimPlayer::uidCounter = 0;

list<AnimPlayer::AnimElement*> AnimPlayer::m_table;



/** Inits the anim tables
 */
void AnimPlayer::init()
{
  cleanup();
}



/** Cleans up the animation tables
 */
void AnimPlayer::cleanup()
{
  list< AnimElement* >::iterator it = m_table.begin();
  
  while ( it != m_table.end() ) {
    if ( (*it) != 0 ) {
      delete (*it);
    }
    it++;
  }
  m_table.clear();
}



/** Adds new animation to playlist
 */
int AnimPlayer::spawn( const Animation& a, const Vec2D& p, int t )
{
  AnimElement* pE = new AnimElement;
  pE->iType = t;
  pE->iAnim = a;
  pE->iPos = p;
  pE->iMovement = Vec2D(0,0);
  pE->iDrawmode = DMODE_NORMAL;
  pE->iAlpha = 0;
  pE->iUid = uidCounter++;

  m_table.push_back( pE );
  return pE->iUid;
}


/** Spawns new moving animation.
 */
int AnimPlayer::spawn( const Animation& a, const Vec2D& p, const Vec2D& m,
                        int t )
{
  AnimElement* pE = new AnimElement;
  pE->iType = t;
  pE->iAnim = a;
  pE->iPos = p;
  pE->iMovement = m;
  pE->iDrawmode = DMODE_NORMAL;
  pE->iAlpha = 0;
  pE->iUid = uidCounter++;
  
  m_table.push_back( pE );
  return pE->iUid;
}


/** Adds new lighteffect animation to playlist
 */
int AnimPlayer::spawnLit( const Animation& a, const Vec2D& p, int t )
{
  AnimElement* pE = new AnimElement;
  pE->iType = t;
  pE->iAnim = a;
  pE->iPos = p;
  pE->iMovement = Vec2D(0,0);
  pE->iDrawmode = DMODE_DODGE;
  pE->iAlpha = 50;
  pE->iUid = uidCounter++;

  m_table.push_back( pE );
  return pE->iUid;
}



/** Adds new animation with parameters
 */
int AnimPlayer::spawn( const Animation& rAnim, const Vec2D& rP,
                        DrawMode dmode, int alpha, const Vec2D& rMov )
{
  AnimElement* pE = new AnimElement;
  pE->iType = 0;
  pE->iAnim = rAnim;
  pE->iPos = rP;
  pE->iMovement = rMov;
  pE->iDrawmode = dmode;
  pE->iAlpha = alpha;
  pE->iUid = uidCounter++;
  
  m_table.push_back( pE );
  return pE->iUid;
}


/** Updates the animations and removes them once they've been played
 * through.
 */
void AnimPlayer::update()
{
  list<AnimElement*>::iterator it = m_table.begin();
  while ( it != m_table.end() ) {
    AnimElement* pE = *it;
    pE->iPos += pE->iMovement;
    pE->iAnim.update();
    if ( pE->iAnim.paused() ) {
      animationEndingActions( pE );
      delete pE;
      it = m_table.erase( it );
    } else {
      it++;
    }
  }
}



/** Draws the animations on given bitmap.
 */
void AnimPlayer::redraw( BITMAP* pB )
{
  list<AnimElement*>::iterator it = m_table.begin();
  while ( it != m_table.end() ) {
    AnimElement* pE = *it;
    if ( pE != 0 ) {
      AnimFrame f = pE->iAnim.currentFrame();
      RLE_SPRITE* pS = f.asRleSprite();
      
      int x = pE->iPos.intX() - ( pS->w / 2 );
      int y = pE->iPos.intY() - ( pS->h / 2 );
      x -= Map::scrollX;
      y -= Map::scrollY;

      if ( f.iFrameBlend == ESolid ) {
        draw_rle_sprite( pB, f.asRleSprite(), x, y );
      } else {
        eng2d::SetBlendFunction( f.iFrameBlend, f.iFrameAlpha );
        draw_trans_rle_sprite( pB, f.asRleSprite(), x, y );
      }
    }
    it++;
  }
}



/** Returns the pointer to index'th animation
 */
AnimPlayer::AnimElement* AnimPlayer::getAnim(int index)
{
  if (index >= m_table.size()) {
    return 0;
  }
  list<AnimElement*>::iterator it = m_table.begin();
  for (int i=0; i < index; i++ ) {
    it++;
  }
  return *it;
}


/** FInds the animation whose uid is given. 
 */
AnimPlayer::AnimElement* AnimPlayer::findAnim( int aUid )
{
  list<AnimElement*>::iterator it = m_table.begin();
  for ( it = m_table.begin(); it != m_table.end(); it++ ) {
    if ( (*it)->iUid == aUid ) {
      return *it;
    }
  }
  return 0;
}


/** Removes an animation from the anima table.
 */
void AnimPlayer::removeAnimation( int aUid )
{
  list<AnimElement*>::iterator it = m_table.begin();
  for ( it = m_table.begin(); it != m_table.end(); it++ ) {
    if ( (*it)->iUid == aUid ) {
      m_table.erase( it );
      return;
    }
  }
}


/** Does the animation ending actions
 */
void AnimPlayer::animationEndingActions( AnimElement* aAnimElement )
{
  switch ( aAnimElement->iType ) {
    case ( EBulletShell ): {
      Sound::playSample( SMP_BULLETSHELL, false );
      break;
    }
  }
}

} // end of namespace

